/*
 CryptoCC.cpp
 Wrapper library around Crypto++ PKCS5_PBKDF2_HMAC functions for use with OTFBrutus

 Copyright (c) 2010 Josh Harris (tateu)

 Redistribution and use in source and binary forms, with or without modification,
 are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice,
     this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright notice,
     this list of conditions and the following disclaimer in the documentation and/or
     other materials provided with the distribution.
  3. The name of the author may not be used to endorse or promote products derived
     from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE
 LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

	2010-12-11 - JHH
		* v0.1.0.0 - Original release
*/

#include <windows.h>

#include "CryptoCC.h"

#include "config.h"
#include "cpu.h"
#include "ripemd.h"
#include "sha.h"
#include "whrlpool.h"
#include "pwdbased.h"

USING_NAMESPACE(CryptoPP)
USING_NAMESPACE(std)

#pragma comment(lib,"cryptlib")

/*extern "C" */
void CC_hmac_truncate
  (
	  char *d1,		/* data to be truncated */
	  char *d2,		/* truncated data */
	  int len		/* length in bytes to keep */
)
{
	int i;
	for (i = 0; i < len; i++)
		d2[i] = d1[i];
}

void CryptoPP_PKCS5_PBKDF2_HMAC(char *pwd, int pwd_len, char *salt, int salt_len, int iterations, char *dk, int dklen, int hash)
{
	PKCS5_PBKDF2_HMAC<RIPEMD160> pbkdf_rmd160;
	PKCS5_PBKDF2_HMAC<SHA512> pbkdf_sha512;
	PKCS5_PBKDF2_HMAC<Whirlpool> pbkdf_whirlpool;
	PKCS5_PBKDF2_HMAC<SHA1> pbkdf_sha1;

    if (hash == 1)
		//CC_derive_key(pwd, pwd_len, salt, salt_len, iterations, dk, dklen, hash);
		pbkdf_rmd160.DeriveKey((byte *)dk, dklen, 0, (byte *)pwd, pwd_len, (byte *)salt, salt_len, iterations);
	else if (hash == 2)
		//CC_derive_key(pwd, pwd_len, salt, salt_len, iterations, dk, dklen, hash);
		pbkdf_sha512.DeriveKey((byte *)dk, dklen, 0, (byte *)pwd, pwd_len, (byte *)salt, salt_len, iterations);
	else if (hash == 3)
		//CC_derive_key(pwd, pwd_len, salt, salt_len, iterations, dk, dklen, hash);
		pbkdf_whirlpool.DeriveKey((byte *)dk, dklen, 0, (byte *)pwd, pwd_len, (byte *)salt, salt_len, iterations);
	else
		//CC_derive_key(pwd, pwd_len, salt, salt_len, iterations, dk, dklen, hash);
		pbkdf_sha1.DeriveKey((byte *)dk, dklen, 0, (byte *)pwd, pwd_len, (byte *)salt, salt_len, iterations);

	/*if (hash == 1)
		pbkdf_rmd160.DeriveKey((byte *)dk, dklen, 0, (byte *)pwd, pwd_len, (byte *)salt, salt_len, iterations);
	else if (hash == 2)
		CC_derive_key(pwd, pwd_len, salt, salt_len, iterations, dk, dklen, hash);
	else if (hash == 3)
		CC_derive_key(pwd, pwd_len, salt, salt_len, iterations, dk, dklen, hash);
	else
		pbkdf_sha1.DeriveKey((byte *)dk, dklen, 0, (byte *)pwd, pwd_len, (byte *)salt, salt_len, iterations);*/
}

#if 0
///////////////////////////////////////////////////
///////////////////////////////////////////////////
//
/*
 Legal Notice: Some portions of the source code contained in this file were
 derived from the source code of Encryption for the Masses 2.02a, which is
 Copyright (c) 1998-2000 Paul Le Roux and which is governed by the 'License
 Agreement for Encryption for the Masses'. Modifications and additions to
 the original source code (contained in this file) and all other portions
 of this file are Copyright (c) 2003-2009 TrueCrypt Developers Association
 and are governed by the TrueCrypt License 3.0 the full text of which is
 contained in the file License.txt included in TrueCrypt binary and source
 code distribution packages. */
 
// Modified from TrueCrypt Pkcs5.c
//

void CC_hmac_init
(
	  char *k,		/* secret key */
	  int lk,		/* length of the key in bytes */
	  HashTransformation *t_ictx,
	  HashTransformation *t_octx
)
{
	char *key;
	char *buf;
	int i;
	int HASH_DIGESTSIZE = t_ictx->DigestSize();
	int HASH_BLOCKSIZE = t_ictx->BlockSize();

	key = new char[HASH_DIGESTSIZE];
	buf = new char[HASH_BLOCKSIZE];

	/* If the key is longer than the hash algorithm block size,
	   let key = HashTransformation(key), as per HMAC specifications. */
	if (lk > HASH_BLOCKSIZE)
	{
		t_ictx->CalculateDigest((byte *)key, (byte *)k, lk);
		k = key;
		lk = HASH_DIGESTSIZE;
	}

	// Pad the key for inner digest
	for (i = 0; i < lk; ++i)
		buf[i] = (char) (k[i] ^ 0x36);
	for (i = lk; i < HASH_BLOCKSIZE; ++i)
		buf[i] = 0x36;

	t_ictx->Update((byte *)buf, HASH_BLOCKSIZE);
	
	for (i = 0; i < lk; ++i)
		buf[i] = (char) (k[i] ^ 0x5C);
	for (i = lk; i < HASH_BLOCKSIZE; ++i)
		buf[i] = 0x5C;

	t_octx->Update((byte *)buf, HASH_BLOCKSIZE);

	delete key;
	delete buf;
}
//*JHH

void CC_hmac
(
	  char *d,		/* data */
	  int ld,		/* length of data in bytes */
	  char *out,		/* output buffer, at least "t" bytes */
	  int t,
	  HashTransformation *t_ictx, //JHH
	  HashTransformation *t_octx //JHH
)
{
	HashTransformation *pctx;
	int HASH_DIGESTSIZE = t_ictx->DigestSize();
	int HASH_BLOCKSIZE = t_ictx->BlockSize();

	pctx = (HashTransformation *)t_ictx->Clone();
	pctx->Update((byte *)d, ld);
	pctx->Final((byte *)out);

	delete pctx;

	//memcpy (&ictx, &t_ictx, sizeof(t_ictx)); //JHH
	//ictx.Update((byte *)d, ld);
	//ictx.Final((byte *)out);

	// Outer Digest
	pctx = (HashTransformation *)t_octx->Clone();
	pctx->Update((byte *)out, HASH_DIGESTSIZE);
	pctx->Final((byte *)out);
	//memcpy (&octx, &t_octx, sizeof(t_octx)); //JHH
	//ictx.Update((byte *)out, HASH_DIGESTSIZE);
	//octx.Final((byte *)out);

	delete pctx;
}

void CC_derive_u (char *salt, int salt_len, int iterations, char *u, int b, HashTransformation *t_ictx, HashTransformation *t_octx)
{
	char *j;
	char *k;
	char init[128];
	char counter[4];
	int c, i;
	int HASH_DIGESTSIZE = t_ictx->DigestSize();

	j = new char[HASH_DIGESTSIZE];
	k = new char[HASH_DIGESTSIZE];

	// iteration 1
	memset (counter, 0, 4);
	counter[3] = (char) b;
	memcpy (init, salt, salt_len);	// salt
	memcpy (&init[salt_len], counter, 4);	// big-endian block number
	CC_hmac (init, salt_len + 4, j, HASH_DIGESTSIZE, t_ictx, t_octx); //JHH
	memcpy (u, j, HASH_DIGESTSIZE);

	// remaining iterations
	for (c = 1; c < iterations; c++)
	{
		CC_hmac (j, HASH_DIGESTSIZE, k, HASH_DIGESTSIZE, t_ictx, t_octx); //JHH
		for (i = 0; i < HASH_DIGESTSIZE; i++)
		{
			u[i] ^= k[i];
			j[i] = k[i];
		}
	}

	delete j;
	delete k;
}

void CC_derive_key (char *pwd, int pwd_len, char *salt, int salt_len, int iterations, char *dk, int dklen, int hash)
{
	HashTransformation *t_ictx = NULL;
	HashTransformation *t_octx = NULL;
	SHA1 i_sha1, o_sha1;
	SHA512 i_sha512, o_sha512;
	RIPEMD160 i_rmd160, o_rmd160;
	Whirlpool i_whr, o_whr;
	char *u;
	int b, l, r;

	int HASH_DIGESTSIZE;

	if (hash == 1) {
		t_ictx = &i_rmd160;
		t_octx = &o_rmd160;
	} else if (hash == 2) {
		t_ictx = &i_sha512;
		t_octx = &o_sha512;
	} else if (hash == 3) {
		t_ictx = &i_whr;
		t_octx = &o_whr;
	} else if (hash == 4) {
		t_ictx = &i_sha1;
		t_octx = &o_sha1;
	}

	HASH_DIGESTSIZE = t_ictx->DigestSize();
	u = new char[HASH_DIGESTSIZE];

	if (dklen % HASH_DIGESTSIZE)
	{
		l = 1 + dklen / HASH_DIGESTSIZE;
	}
	else
	{
		l = dklen / HASH_DIGESTSIZE;
	}

	r = dklen - (l - 1) * HASH_DIGESTSIZE;

    CC_hmac_init(pwd, pwd_len, t_ictx, t_octx);
	
	// first l - 1 blocks
	for (b = 1; b < l; b++)
	{
		CC_derive_u (salt, salt_len, iterations, u, b, t_ictx, t_octx);
		memcpy (dk, u, HASH_DIGESTSIZE);
		dk += HASH_DIGESTSIZE;
	}

	/* last block */
	CC_derive_u (salt, salt_len, iterations, u, b, t_ictx, t_octx);
	memcpy (dk, u, r);

	delete u;
}
#endif
