/***************************************************************
 * Name:      OTFBrutus.cpp
 * Author:    Josh Harris (tateu@tateu.net)
 * Created:   2010-12-01
 * Copyright: Josh Harris (www.tateu.net)
 * License:
 Redistribution and use in source and binary forms, with or without modification,
 are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice,
     this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright notice,
     this list of conditions and the following disclaimer in the documentation and/or
     other materials provided with the distribution.
  3. The name of the author may not be used to endorse or promote products derived
     from this software without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE
 LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 **************************************************************/

#include "OTFBrutus.h"

////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////
//
#include <fstream>
#include <math.h>
#include <process.h>
#include <vector>
#include <algorithm>
#include <string>

#include "Common/Tcdefs.h"
#include "Common/Crypto.h"
#include "Common/KeyFiles.h"
#include "Common/Password.h"
#include "Common/Volumes.h"

#include "Utilities.h"

using namespace std;

static char *hashNames[] =
{
	{"Ripemd160"},
	{"SHA512"},
	{"Whirlpool"},
	{"SHA1"}                        // Deprecated/legacy
};

static char *modeNames[] =
{
	{"XTS"},
	{"LRW"},                       // Deprecated/legacy
	{"CBC"}                        // Deprecated/legacy
};

static char *cipherNames[] =
{
	{"AES"},
	{"Serpent"},
	{"Twofish"},
	{"AES-Twofish"},
	{"AES-Twofish-Serpent"},
	{"Serpent-AES"},
	{"Serpent-Twofish-AES"},
	{"Twofish-Serpent"},
	{"Blowfish"},                   // Deprecated/legacy
	{"CAST5"},                      // Deprecated/legacy
	{"Triple DES"},                 // Deprecated/legacy
	{"AES-Blowfish"},               // Deprecated/legacy
	{"AES-Blowfish-Serpent"},       // Deprecated/legacy
	{"Idea"}                        // Deprecated/legacy
};

typedef struct
{
	BOOL bRunning;
	Password pwd;
	BOOL bBoot;
} THREAD_PARAMS, *PTHREAD_PARAMS;

typedef struct
{
	string charItem;
} CHAR_SET, *PCHAR_SET;

typedef struct
{
	vector<CHAR_SET> charSet;
	vector<int> posCount;
	vector<int> subCurPos;
	vector<int> lengths;
	int minLength;
	int maxLength;
	int curLength;
	int maxRepeat;
	int curPos;
	int parentPos;
} PWD_PARTS, *PPWD_PARTS;

bool isRunning;

char *OTFFile = NULL;
char *passFile = NULL;

char headerCipher[512];
int ctx_hash_idx, ctx_cipher_idx, ctx_mode_idx;
vector<int> user_cipher, user_hash, user_mode;
vector<string> keyFiles;
bool useKeyFiles = false;
KeyFile *firstKeyFile = NULL;
CIPHER_SET cipher_set;
int externalPKCS = 0;

string found_pwd;
INT64 actual_pwds_tried = 0;

INT64 idxStart = 1;
INT64 idxEnd = 0;

DWORD dwEvent = 0;
int iVolType = 0;
bool useBackupHeader = false;
BOOL bSuccess = FALSE;
char *pwd_pattern = NULL, *dictionary = NULL;
size_t pwd_pattern_len = 0;
INT64 pwd_count = 0, pwd_count_dict = 0, print_count = 0;
INT64 pwd_idx = 0, pwd_idx_dict = 0;
INT64 counter = 0;
INT64 counterSkipped = 0;
INT64 pwd_count_total = 0;

string pwd;
vector<PWD_PARTS> m_vpwdParts;

INT64 dictionaryStartPos;

DWORD size_MB;

BOOL bVerbose = FALSE;
BOOL bSaveWordList = FALSE;
BOOL bOverwrite = FALSE;
wxString wordListFile;
BOOL bComplete = FALSE;
int threads = 0;

HANDLE hEvent[33];
HANDLE hEventThread[33];
HANDLE hThread[33];
THREAD_PARAMS thread_params[32];

HANDLE hRunEvent;
HANDLE hRunThread;

wxString err, logString, status;

extern HWND hwndMain;
//
////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////
// Threads
string ParsePart(PWD_PARTS *pwdPart)
{
	string pwd(_T(""));
	int curPos = 0;
	bool skip = false;

	for (int i = 0; i < pwdPart->posCount.size(); i++) {
		pwdPart->posCount[i] = 0;
	}

	int curLength = pwdPart->lengths[pwdPart->curLength];

	for (int i = 0; i < curLength; i++) {
		curPos = pwdPart->subCurPos[i];
		if (pwdPart->maxRepeat > 0 && pwdPart->posCount[curPos] >= pwdPart->maxRepeat) {
			skip = true;
			break;
		} else {
			pwd.append(pwdPart->charSet[curPos].charItem);
			pwdPart->posCount[curPos]++;
		}
	}

	if (skip) {
		pwd = _T("");
	}

	return pwd;
}

void passwordThread(void *arg)
{
	int i;
	ofstream stream;

	if (bSaveWordList) {
		pwd.clear();
		if (bOverwrite)
			stream.open(wordListFile.c_str());
		else
			stream.open(wordListFile.c_str(), ios_base::app);
		if (!stream.is_open()) {
			bSuccess = TRUE;
			err.Printf(wxT("ERROR Can't open word list:\n  %s\n"), wordListFile.c_str());
		}
	}

	for (pwd_idx = 0; pwd_idx < idxEnd && !bSuccess && isRunning; pwd_idx++) {
		if (idxStart <= pwd_idx && !bSaveWordList) {
			pwd.clear();
		}

		bool skip = false;
		string pwdPart(_T(""));
		for (int i = 0; i < m_vpwdParts.size(); i++) {
			if (m_vpwdParts[i].lengths[m_vpwdParts[i].curLength] == 0) {
				continue;
			}

			string s = ParsePart(&m_vpwdParts[i]);
			if (s.empty()) {
				skip = true;
				break;
			} else {
				pwdPart.append(s);
			}
		}

		if (skip || pwdPart.size() <= 0) {
			counterSkipped++;
			goto Skip_Count;
		} else {
			pwd.append(pwdPart);

			if (idxStart <= pwd_idx) {
				if (bSaveWordList) {
					if (pwd.at(pwd.size() - 1) == '\n') {
						counterSkipped++;
						goto Skip_Count;
					}
					//if (pwd.at(pwd.size() - 1) != '\n')
						pwd.append("\n");
					if (pwd.size() >= 1048576) {
						stream.write(pwd.c_str(), pwd.size());
						pwd.clear();
					}
				} else {
					int id = WAIT_OBJECT_0 + dwEvent;
					if (id >= 0 && id < threads) {
						memcpy(thread_params[id].pwd.Text, pwd.c_str(), pwd.size());
						thread_params[id].pwd.Length = pwd.size();
						thread_params[id].bRunning = TRUE;
						thread_params[id].bBoot = iVolType == OTF_VOL_BOOT ? TRUE : FALSE;
						ResetEvent(hEvent[id]);
						SetEvent(hEventThread[id]);
					} else {
						//printf("\ndwEvent = %d\n", dwEvent);
					}

					dwEvent = WaitForMultipleObjects(
						threads, // number of objects in array
						hEvent,  // array of objects
						FALSE,   // wait for all
						5000);   // INFINITE
				}

				counter++;
			}

			if (bSuccess) {
				bComplete = TRUE;
				break;
			} else if (!isRunning) {
				bComplete = TRUE;
				break;
			}
		}

Skip_Count:
		// increment the chars from last position first
		for (int i = m_vpwdParts.size() - 1; i >= 0; i--) {
			if (m_vpwdParts[i].lengths[m_vpwdParts[i].curLength] == 0) {
				m_vpwdParts[i].curLength++;
			} else {
				// increment the chars from last position first
				for (int j = m_vpwdParts[i].lengths[m_vpwdParts[i].curLength] - 1; j >= 0; j--) {
					m_vpwdParts[i].subCurPos[j]++;

					if (m_vpwdParts[i].subCurPos[j] >= m_vpwdParts[i].charSet.size()) {
						m_vpwdParts[i].subCurPos[j] = 0;
						if (j == 0) {
							m_vpwdParts[i].curLength++;
						}
					} else {
						break;
					}
				}
			}

			if (m_vpwdParts[i].curLength >= m_vpwdParts[i].lengths.size()) {
				m_vpwdParts[i].curLength = 0;
			} else {
				break;
			}
		}
	}

	if (bSaveWordList) {
		if (stream.is_open()) {
			if (pwd.size()) {
				stream.write(pwd.c_str(), pwd.size());
				pwd.clear();
			}
			stream.close();
		}
	}

	SetEvent(hEvent[threads]);
	SetEvent(hEventThread[threads]);
}

void dictionaryThread(void *arg)
{
	ifstream stream;
	stream.open(dictionary);
	if (stream.is_open()) {
		if (dictionaryStartPos != 0) {
			stream.seekg(dictionaryStartPos, ios::beg);
			pwd_idx_dict = idxStart;
		}

		while (getline(stream, pwd) && !bSuccess && isRunning && pwd_idx_dict < idxEnd) {
			//bSuccess = OTFE_Decrypt_Header(ciphertext, plaintext, 512, (byte *)pwd.c_str(), pwd.size());

			if (idxStart <= pwd_idx_dict) {
				int id = WAIT_OBJECT_0 + dwEvent;
				if (id >= 0 && id < threads) {
					memcpy(thread_params[id].pwd.Text, pwd.c_str(), pwd.size());
					thread_params[id].pwd.Length = pwd.size();
					thread_params[id].bRunning = TRUE;
					thread_params[id].bBoot = iVolType == OTF_VOL_BOOT ? TRUE : FALSE;
					ResetEvent(hEvent[id]);
					SetEvent(hEventThread[id]);
				} else {
					//printf("\ndwEvent = %d\n", dwEvent);
				}

				dwEvent = WaitForMultipleObjects(
					threads, // number of objects in array
					hEvent,  // array of objects
					FALSE,   // wait for all
					5000);   // INFINITE

				counter++;
			}

			if (bSuccess) {
				bComplete = TRUE;
				break;
			} else if (!isRunning) {
				bComplete = TRUE;
				break;
			}

			pwd_idx_dict++;
		}
	} else {
		err.Printf(wxT("ERROR Can't open dictionary:\n  %s\n"), dictionary);
		PostMessage(hwndMain, OTF_ERROR, 0, 0);
		_endthread();
	}

	stream.close();

	SetEvent(hEvent[threads]);
	SetEvent(hEventThread[threads]);
	_endthread();
}

void encryptionThread(void *arg)
{
	int nStatus = 3;
	DWORD dwWaitResult;
	PCRYPTO_INFO cryptoInfo = NULL;
	int id = (INT_PTR)arg;
	string temp_pwd;

start:
	dwWaitResult = WaitForSingleObject(hEventThread[id], INFINITE);
	if (dwWaitResult == WAIT_OBJECT_0)
	{
		if (!bComplete && isRunning) {
			if (thread_params[id].bRunning) {
				ResetEvent(hEventThread[id]);

				thread_params[id].pwd.Text[thread_params[id].pwd.Length] = 0;
				temp_pwd = reinterpret_cast<const char*>(thread_params[id].pwd.Text);
				if (firstKeyFile && useKeyFiles) {
					KeyFilesApply (&thread_params[id].pwd, firstKeyFile);
				}

				//if (bVerbose) {
				//	printf("Password: %s\n", thread_params[id].pwd.Text);
				//}

				nStatus = ReadVolumeHeader (iVolType == OTF_VOL_BOOT ? TRUE : FALSE, headerCipher, &thread_params[id].pwd, &cryptoInfo, NULL, &cipher_set); //&cryptoInfo, cryptoInfo
				if (nStatus == ERR_CIPHER_INIT_WEAK_KEY) {
					nStatus = 0;	// We can ignore this error here
				}

				actual_pwds_tried++;

				/*if (nStatus == ERR_PASSWORD_WRONG) {
				} else if (nStatus != 0) {
				}*/
				if (nStatus == 0) {
					found_pwd = temp_pwd;
					bSuccess = bComplete = TRUE;
					ctx_hash_idx = cryptoInfo->pkcs5 - FIRST_PRF_ID;
					ctx_mode_idx = cryptoInfo->mode - FIRST_MODE_OF_OPERATION_ID;
					ctx_cipher_idx = cryptoInfo->ea - EAGetFirst ();
					for (int i = 0; i <= threads; i++) {
						SetEvent(hEvent[i]);
						SetEvent(hEventThread[i]);
					}
					_endthread();
				}

				SetEvent(hEvent[id]);
			}
		} else {
//			ResetEvent(hEvent[id]);
			//ResetEvent(hEventThread[id]);
			_endthread();
		}
	}

	if (bComplete || !isRunning) {
		//ResetEvent(hEventThread[id]);
		_endthread();
	}

	goto start;
}
//
////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////
//
void runThread(void *arg)
{
	HANDLE m_hVolume;
	DWORD dwVolAccess = GENERIC_READ;
	DWORD dwBytesRead = 0;
	ULARGE_INTEGER m_i64FileSz;
	LARGE_INTEGER m_iPos;
	char durationStr[60];
	char timeLeftStr[60];
	int i;
	wxString s;

	LARGE_INTEGER nFreq, nStart, nEnd;

	PWD_PARTS pwdParts;
	CHAR_SET charSet;

	bSuccess = FALSE;
	bComplete = FALSE;

	actual_pwds_tried = 0;
	pwd_count = 0;
	pwd_count_dict = 0;
	print_count = 0;
	pwd_idx = 0;
	pwd_idx_dict = 0;
	counter = 0;
	counterSkipped = 0;
	pwd_count_total = 0;

	// Make sure threads is between 1 and 32
	if (threads < 1)
		threads = 1;
	else if (threads > 32)
		threads = 32;

	if (!bSaveWordList) {
		m_hVolume = CreateFile(OTFFile, dwVolAccess,
					FILE_SHARE_READ | FILE_SHARE_WRITE, NULL,
					OPEN_EXISTING, FILE_FLAG_NO_BUFFERING | FILE_ATTRIBUTE_READONLY,
					NULL);

		if (m_hVolume == INVALID_HANDLE_VALUE) {
			LPVOID lpMsgBuf;

			FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
						  NULL,
						  GetLastError(),
						  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
						  (LPTSTR) &lpMsgBuf,
						  0,
						  NULL);

			err.Printf(wxT("Error loading volume\n%s\n%s\n"), (LPCTSTR)lpMsgBuf, OTFFile);

			LocalFree(lpMsgBuf);
			//SecureZeroMemory(szPwd, sizeof(szPwd));
			goto END;
		}

		m_i64FileSz.LowPart = GetFileSize(m_hVolume, &m_i64FileSz.HighPart);
		m_iPos.QuadPart = 0;

		if (iVolType == OTF_VOL_BOOT)
		{
			user_hash.clear();
			user_hash.push_back(OTF_RIPEMD160); //The only hash allowed for system encryption as of TC 7.0a
			user_mode.clear();
			user_mode.push_back(OTF_XTS); //The only mode allowed for system encryption as of TC 7.0a

			if (m_i64FileSz.QuadPart < BOOT_ISO_OFFSET) {
				m_iPos.QuadPart = 0; //Assume 512 byte Header only
			} else {
				if (m_i64FileSz.QuadPart == ISO_SIZE) {
					m_iPos.QuadPart = BOOT_ISO_OFFSET; //Rescue Disc
				} else {
					m_iPos.QuadPart = 62 * TC_SECTOR_SIZE_BIOS; //Device
				}
			}
		}
		else if (iVolType)
		{
			/*if (m_i64FileSz.QuadPart <= BACKUP_H_SIZE_LEGACY)
				m_iPos.QuadPart = -TC_VOLUME_HEADER_SIZE_LEGACY; //Old TC Header format, backup header file is 1024 bytes
			else if (m_i64FileSz.QuadPart == BACKUP_H_SIZE)
				m_iPos.QuadPart = TC_HIDDEN_VOLUME_HEADER_OFFSET; //New Header Format, backup header file is 131072 bytes
			else */if (iVolType == OTF_VOL_HIDDEN_LEGACY1)
				m_iPos.QuadPart = -TC_HIDDEN_VOLUME_HEADER_OFFSET_LEGACY; //File or partition, Legacy, TrueCrypt version prior to 6.0
			else
				m_iPos.QuadPart = TC_HIDDEN_VOLUME_HEADER_OFFSET; //File or partition
		}

		if (useBackupHeader && iVolType != OTF_VOL_BOOT && iVolType != OTF_VOL_HIDDEN_LEGACY1) {
			if (m_i64FileSz.QuadPart < TC_VOLUME_HEADER_GROUP_SIZE * 2) {
				err.Printf(wxT("Volume is too small, cannot use embedded backup header\n"));
				CloseHandle(m_hVolume);
				goto END;
			}
			//Use embedded backup header
			m_iPos.QuadPart = iVolType == OTF_VOL_HIDDEN ?  -TC_HIDDEN_VOLUME_HEADER_OFFSET : -TC_VOLUME_HEADER_GROUP_SIZE;
		}

		if (m_iPos.QuadPart != 0) {
			int seekStart = m_iPos.QuadPart > 0 ? FILE_BEGIN : FILE_END;
			if (!SetFilePointer(m_hVolume, m_iPos.LowPart, &m_iPos.HighPart, seekStart)) {
				err.Printf(wxT("Cannot seek to volume header position\n"));
				CloseHandle(m_hVolume);
				goto END;
			}
		}

		if (!ReadFile(m_hVolume, headerCipher, TC_SECTOR_SIZE_LEGACY, &dwBytesRead, FALSE)) {
			err.Printf(wxT("Cannot read volume header\n"));
			CloseHandle(m_hVolume);
			goto END;
		}

		CloseHandle(m_hVolume);
	}

	/////////////////////////////////
	//
	m_vpwdParts.clear();

	if (pwd_pattern == NULL) {
		pwd_pattern_len = 0;
		pwd_count = 0;
	} else {
		pwd_pattern_len = strlen(pwd_pattern);

		for (int i = 0; i < pwd_pattern_len; i++) {
			bool isClass = false;
			bool isEscaped = false;
			int k = 0;
			int closeBraceAt = -1;
			int minLength = 1;
			int maxLength = 1;
			int pattern_dupes = 0;
			pwdParts.charSet.clear();
			pwdParts.posCount.clear();
			pwdParts.subCurPos.clear();
			pwdParts.lengths.clear();
			charSet.charItem.clear();

			if (i > 0) {
				if (	pwd_pattern[i-1] != '\\' &&
					(
						pwd_pattern[i]   == ']' ||
						pwd_pattern[i]   == ')' ||
						pwd_pattern[i]   == '{' ||
						pwd_pattern[i]   == '}' ||
						pwd_pattern[i]   == '|'
					)
				) {
					err.Printf(wxT("ERROR at pos %d: found invalid character '%c' not escaped with backslash\n"), i, pwd_pattern[i]);
					goto END;
				}
			} else {
				if (	pwd_pattern[i]   == ']' ||
						pwd_pattern[i]   == ')' ||
						pwd_pattern[i]   == '{' ||
						pwd_pattern[i]   == '}' ||
						pwd_pattern[i]   == '|'
				) {
					err.Printf(wxT("ERROR at pos %d: found invalid character '%c' not escaped with backslash\n"), i, pwd_pattern[i]);
					goto END;
				}
			}
			
			if (pwd_pattern[i] == '[') {
				isClass = true;
				// increase pos counter to move past the opening '['
				i++;

				isEscaped = false;

				//Find end bracket
				while (i + k < pwd_pattern_len) {
					if (pwd_pattern[i + k] == '\\' && !isEscaped) {
						isEscaped = true;
					} else if (pwd_pattern[i + k] == ']' && !isEscaped) {
						closeBraceAt = k + i;
						k = 0;
						break;
					} else {
						isEscaped = false;
					}
					k++;
				}

				if (closeBraceAt == -1) {
					err.Printf(wxT("ERROR at pos %d: did not find closing ']'\n"), i);
					goto END;
				}

				isEscaped = false;

				while (i < closeBraceAt) {
					charSet.charItem.clear();

					if (	!isEscaped &&
							(
								pwd_pattern[i]   == '[' ||
								pwd_pattern[i]   == ']' ||
								pwd_pattern[i]   == '(' ||
								pwd_pattern[i]   == ')' ||
								pwd_pattern[i]   == '{' ||
								pwd_pattern[i]   == '}' ||
								pwd_pattern[i]   == ':' ||
								pwd_pattern[i]   == '|'
							)
						) {
							err.Printf(wxT("ERROR at pos %d: found invalid character '%c' not escaped with backslash\n"), i, pwd_pattern[i]);
							goto END;
					} else if (pwd_pattern[i] == '\\' && !isEscaped) {
						isEscaped = true;
						i++;
						continue;
					} else if (pwd_pattern[i + 1] == '-' && !isEscaped) {
						char s = pwd_pattern[i];
						i += 2;

						if (i >= pwd_pattern_len - 1) {
							err.Printf(wxT("ERROR at pos %d: found range '-' char without valid End Char\n"), i);
							goto END;
						}

						if (pwd_pattern[i] == ']') {
							err.Printf(wxT("ERROR at pos %d: found range '-' char without valid End Char\n"), i);
							goto END;
						}

						if (pwd_pattern[i] == '\\') {
							i++;
						}

						if (i >= pwd_pattern_len - 1) {
							err.Printf(wxT("ERROR at pos %d: found range '-' char without valid End Char\n"), i);
							goto END;
						}

						if (s < pwd_pattern[i]) {
							for (char j = s; j <= pwd_pattern[i]; j++) {
								charSet.charItem.clear();
								charSet.charItem.push_back(j);
								pwdParts.charSet.push_back(charSet);
							}
						} else {
							for (char j = s; j >= pwd_pattern[i]; j--) {
								charSet.charItem.clear();
								charSet.charItem.push_back(j);
								pwdParts.charSet.push_back(charSet);
							}
						}

						isEscaped = false;
						i++;
						continue;
					}

					charSet.charItem.push_back(pwd_pattern[i]);
					pwdParts.charSet.push_back(charSet);

					isEscaped = false;
					i++;
				} // END while (i < closeBraceAt) {

				// increase pos counter to move past the closing ']'
				i++;
			} else if (pwd_pattern[i] == '(') { // END if (pwd_pattern[i] == '[') {
				isClass = true;
				// increase pos counter to move past the opening '('
				i++;

				isEscaped = false;

				//Find end bracket
				while (i + k < pwd_pattern_len) {
					if (pwd_pattern[i + k] == '\\' && !isEscaped) {
						isEscaped = true;
					} else if (pwd_pattern[i + k] == ')' && !isEscaped) {
						closeBraceAt = k + i;
						k = 0;
						break;
					} else {
						isEscaped = false;
					}
					k++;
				}

				if (closeBraceAt == -1) {
					err.Printf(wxT("ERROR at pos %d: did not find closing ')'\n"), i);
					goto END;
				}

				charSet.charItem.clear();
				isEscaped = false;

				while (i < closeBraceAt) {
					if (	!isEscaped &&
							(
								pwd_pattern[i]   == '[' ||
								pwd_pattern[i]   == ']' ||
								pwd_pattern[i]   == '(' ||
								pwd_pattern[i]   == ')' ||
								pwd_pattern[i]   == '{' ||
								pwd_pattern[i]   == '}' ||
								pwd_pattern[i]   == ':' ||
								pwd_pattern[i]   == '-'
							)
						) {
							err.Printf(wxT("ERROR at pos %d: found invalid character '%c' not escaped with backslash\n"), i, pwd_pattern[i]);
							goto END;
					} else if (pwd_pattern[i] == '\\' && !isEscaped) {
						isEscaped = true;
						i++;
						continue;
					} else if (pwd_pattern[i] == '|' && !isEscaped) {
						i++;
						if (charSet.charItem.size() > 0) {
							pwdParts.charSet.push_back(charSet);
						}

						charSet.charItem.clear();
						isEscaped = false;
						continue;
					}

					charSet.charItem.push_back(pwd_pattern[i]);
					
					isEscaped = false;
					i++;
				} // END while (i < closeBraceAt) {

				if (charSet.charItem.size() > 0) {
					pwdParts.charSet.push_back(charSet);
				} else {
					err.Printf(wxT("ERROR at pos %d: missing string data inside of (), last char '%c'\n"), i-1, pwd_pattern[i-1]);
					goto END;
				}

				// increase pos counter to move past the closing ')'
				i++;
			} else { // END } else if (pwd_pattern[i] == '(') {
				isClass = false;
				isEscaped = false;
				while (i < pwd_pattern_len) {
					if (	!isEscaped &&
							(
								//pwd_pattern[i]   == '[' ||
								pwd_pattern[i]   == ']' ||
								//pwd_pattern[i]   == '(' ||
								pwd_pattern[i]   == ')' ||
								pwd_pattern[i]   == '{' ||
								pwd_pattern[i]   == '}' ||
								pwd_pattern[i]   == ':' ||
								pwd_pattern[i]   == '-'
							)
						) {
							err.Printf(wxT("ERROR at pos %d: found invalid character '%c' not escaped with backslash\n"), i, pwd_pattern[i]);
							goto END;
					} else if (pwd_pattern[i] == '\\' && !isEscaped) {
						isEscaped = true;
						i++;
						continue;
					} else if (pwd_pattern[i-1] != '\\' && (pwd_pattern[i] == '[' || pwd_pattern[i] == '(')) {
						i--; // decrease counter so the for loop finds the char
						break;
					}

					isEscaped = false;
					charSet.charItem.push_back(pwd_pattern[i]);

					i++;
				}

				if (charSet.charItem.size() > 0) {
					pwdParts.charSet.push_back(charSet);
				}
			}

			// Check for length character brace open '{'
			if (isClass && i < pwd_pattern_len) {
				k = 0;
				closeBraceAt = -2;

				if (pwd_pattern[i] == '{') {
					// increase pos counter to move past the opening '{'
					i++;

					isEscaped = false;

					//Find end bracket
					while (i + k < pwd_pattern_len) {
						if (pwd_pattern[i + k] == '\\' && !isEscaped) {
							isEscaped = true;
						} else if (pwd_pattern[i + k] == '}' && !isEscaped) {
							closeBraceAt = k + i;
							k = 0;
							break;
						} else {
							isEscaped = false;
						}
						k++;
					}

					if (closeBraceAt == -1) {
						err.Printf(wxT("ERROR at pos %d: did not find closing '}'\n"), i);
						goto END;
					}
				} else {
					i--; //step back one so main for loop can pick up next char
				}

				string pCountStr;
				bool inRange = false;
				bool inLimit = false;

				while (i <= closeBraceAt) {
					if (pwd_pattern[i] == ',' || pwd_pattern[i] == ':' || pwd_pattern[i] == '}') {
						if (inLimit && pCountStr.size() > 0) {
							if (pwd_pattern[i] != '}') {
								err.Printf(wxT("ERROR at pos %d: found limit character ':' at incorrect offset\n"), i);
								goto END;
							}
							pattern_dupes = atoi(pCountStr.c_str());
							pCountStr.clear();
						} else if (inLimit) {
							err.Printf(wxT("ERROR at pos %d: found range character ':' without a beginning\n"), i);
							goto END;
						}

						if (inRange && pCountStr.size() > 0) {
							maxLength = atoi(pCountStr.c_str());
							
							for (int j = minLength; j <= maxLength; j++) {
								pwdParts.lengths.push_back(j);
							}

							inRange = false;
						} else if (!inRange && pCountStr.size() > 0) {
							pwdParts.lengths.push_back(atoi(pCountStr.c_str()));
						} else if (!inLimit) {
							err.Printf(wxT("ERROR at pos %d: found range character '%c' without a beginning\n"), i, pwd_pattern[i]);
							goto END;
						}

						if (pwd_pattern[i] == '}') {
							break;
						} else if (pwd_pattern[i] == ':') {
							inLimit = true;
						}

						pCountStr.clear();
					} else if (pwd_pattern[i] == '-') {
						if (pCountStr.size() > 0) {
							minLength = atoi(pCountStr.c_str());
							maxLength = minLength;
							inRange = true;
						} else {
							err.Printf(wxT("ERROR at pos %d: found range character '-' without a beginning\n"), i);
							goto END;
						}

						pCountStr.clear();
					} else if (pwd_pattern[i] < '0' || pwd_pattern[i] > '9') {
						err.Printf(wxT("ERROR at pos %d: found invalid character '%c' in length expression\n"), i, pwd_pattern[i]);
						goto END;
					} else {
						pCountStr.push_back(pwd_pattern[i]);							
					}

					i++;
				}// END while (i <= closeBraceAt) {
			} // END Check for length character brace open '{'

			if (pwdParts.lengths.size() <= 0) {
				pwdParts.lengths.push_back(1);
			} else {
				sort(pwdParts.lengths.begin(), pwdParts.lengths.end());
				vector<int>::iterator it = unique(pwdParts.lengths.begin(), pwdParts.lengths.end());
				pwdParts.lengths.resize(it - pwdParts.lengths.begin());
			}

			pwdParts.minLength = pwdParts.lengths.front();
			pwdParts.maxLength = pwdParts.lengths.back();

			if (pwdParts.lengths[0] == 0) {
				pwdParts.lengths.erase(pwdParts.lengths.begin());
				pwdParts.lengths.push_back(0);
			}

			pwdParts.curPos = 0;
			pwdParts.curLength = 0;
			pwdParts.maxRepeat = pattern_dupes;

			if (pwdParts.minLength < 0) {
				//pwdParts.minLength = 1;
				err.Printf(wxT("ERROR at pos %d: Pattern Min length less than 0\n"), i);
				goto END;
			}

			if (pwdParts.maxLength < pwdParts.minLength) {
				//pwdParts.maxLength = pwdParts.minLength;
				err.Printf(wxT("ERROR at pos %d: Pattern Max length less than Min Length\n"), i);
				goto END;
			}

			if (pwdParts.maxLength == 0) {
				//If max length == 0, there is no point to having this pattern
				pwdParts.charSet.clear();
			}

			for (int i = 0; i < pwdParts.charSet.size(); i++) {
				pwdParts.posCount.push_back(0);
			}

			for (int i = 0; i < pwdParts.maxLength; i++) {
				pwdParts.subCurPos.push_back(0);
			}

			if (pwdParts.charSet.size() > 0) {
				m_vpwdParts.push_back(pwdParts);
			}

			//pwdParts.charSet.clear();
			//pwdParts.posCount.clear();
			//pwdParts.subCurPos.clear();
			//pwdParts.lengths.clear();
		} // END for (int i = 0; i < pwd_pattern_len; i++) {
	} // END if (pwd_pattern == NULL)

	if (m_vpwdParts.size() > 0) {
		pwd_count = 1;
		pwd_count_total = 0;
		int comboCount = 1;

		for (int i = 0; i < m_vpwdParts.size(); i++) {
			comboCount *= m_vpwdParts[i].lengths.size();
			if (m_vpwdParts[i].maxRepeat > 0
						&& m_vpwdParts[i].maxLength > m_vpwdParts[i].charSet.size() * m_vpwdParts[i].maxRepeat) {
				err.Printf(wxT("ERROR: MaxLength > charSet size * maxRepeat\n%18d > %12d * %9d\n"), m_vpwdParts[i].maxLength, m_vpwdParts[i].charSet.size(), m_vpwdParts[i].maxRepeat);
				goto END;
			}
		}

		for (int i = 0; i < comboCount; i++) {
			pwd_count = 1;
			for (int j = 0; j < m_vpwdParts.size(); j++) {
				pwd_count *= pow((double)m_vpwdParts[j].charSet.size(), (double)m_vpwdParts[j].lengths[m_vpwdParts[j].curLength]);
			}

			pwd_count_total += pwd_count;

			// increment the chars from last position first
			for (int j = m_vpwdParts.size() - 1; j >= 0; j--) {
				 m_vpwdParts[j].curLength++;

				 if (m_vpwdParts[j].curLength >= m_vpwdParts[j].lengths.size()) {
					m_vpwdParts[j].curLength = 0;
				} else {
					break;
				}
			}
		}

		pwd_count = pwd_count_total;

		if (idxStart > pwd_count_total) {
			err.Printf(wxT("ERROR: Your Start Index is greater than the password count\n  %I64d > %I64d\n"), idxStart, pwd_count_total);
			goto END;
		}
	}
	//
	/////////////////////////////////

	/////////////////////////////////
	//
	if (!bSaveWordList) {
		logString.append(wxT("Attempting to read the"));
		switch (iVolType) {
			case OTF_VOL_TYPE_STD:
				logString.append(wxT(" Standard"));
				break;
			case OTF_VOL_HIDDEN:
				logString.append(wxT(" Hidden"));
				break;
			case OTF_VOL_BOOT:
				logString.append(wxT(" System"));
				break;
			case OTF_VOL_HIDDEN_LEGACY1:
				logString.append(wxT(" Hidden Legacy"));
				break;
			default:
				logString.append(wxT(" Standard"));
				break;
		}

		logString.append(wxT(" Volume from a"));
		if (_strnicmp(OTFFile, "\\\\?", 3) == 0) {
			logString.append(wxT(" Device"));
		} else {
			logString.append(wxT(" File"));
		}

		logString.append(wxT(" hosted"));
		if (iVolType == OTF_VOL_BOOT) {
			if (m_i64FileSz.QuadPart > ISO_SIZE) {
				logString.append(wxT(" Container"));
			} else {
				logString.append(wxT(" Rescue Disc"));
			}
		} else if (iVolType == OTF_VOL_HIDDEN_LEGACY1) {
			if (m_i64FileSz.QuadPart > BACKUP_H_SIZE_LEGACY) {
				logString.append(wxT(" Container"));
			} else {
				logString.append(wxT(" Header"));
			}
		} else {
			if (m_i64FileSz.QuadPart > BACKUP_H_SIZE) {
				logString.append(wxT(" Container"));
			} else {
				logString.append(wxT(" Header"));
			}
		}
		s.Printf(wxT(":\n  %s\n"), OTFFile);
		logString.append(s);

		if (dictionary != NULL) {
			if (!canAccessFile(dictionary, true, NULL)) {
				err.Printf(wxT("ERROR cannot read dictionary\n  %s\n"), dictionary);
				goto END;
			}
			s.Printf(wxT("Using a custom word list:\n  %s\n"), dictionary);
			logString.append(s);
			logString.append(wxT("  Please wait, counting dictionary list now:"));

			SendMessage(hwndMain, OTF_LOG, 0, 0);
			logString.clear();

			QueryPerformanceFrequency(&nFreq);
			QueryPerformanceCounter(&nStart);

			const int SZ = 1024 * 1024;
			char *temp = new char[SZ];
			int c;

			FILE *f = fopen(dictionary, "rb");
			if (f) {
				INT64 byteCount = 0;
				while (isRunning) {
					c = FileRead(f, temp, SZ);
					if (c <= 0) {
						break;
					}
					byteCount += c;
					pwd_count_dict += CountLines(temp, c, f, byteCount, pwd_count_dict, idxStart, &dictionaryStartPos);
				}
				fclose(f);
			}

			QueryPerformanceCounter(&nEnd);
			double duration = 1.0 * (nEnd.QuadPart - nStart.QuadPart) / nFreq.QuadPart;
			s.Printf(wxT(" %0.2f seconds\n"), duration);
			logString.append(s);

			if (!isRunning) {
				goto END;
			}

			//logString.append(wxT("                                               \r"));
			s.Printf(wxT("Containing:\n  %I64d passwords\n"), pwd_count_dict);
			logString.append(s);

			if (idxStart > pwd_count_dict) {
				err.Printf(wxT("ERROR: Your Start Index is greater than the password count\n  %I64d > %I64d\n"), idxStart, pwd_count_dict);
				goto END;
			}
		}
	} else {
		logString.append(wxT("Saving custom word list to\n  "));
		logString.append(wordListFile);
		logString.append(wxT("\n"));
	}

	if (pwd_count > 0) {
		s.Printf(wxT("Using a password pattern:\n  %s\nContaining:\n  %I64d passwords\n"), pwd_pattern, pwd_count);
		logString.append(s);
	}

	if (!isRunning) {
		goto END;
	}

	if (pwd_count_dict > 0) {
		pwd_count = pwd_count_dict;
		pwd_count_total = pwd_count_dict;
	}

	idxStart--;
	if (idxStart > 0 || idxEnd > 0) {
		if (idxEnd > pwd_count || idxEnd <= 0) {
			idxEnd = pwd_count_total; //pwd_count;
		}
		if (idxStart > idxEnd) {
			idxStart = idxEnd - 1;
		}
		s.Printf(wxT("  Subset Contains:\n    %I64d passwords (%I64d - %I64d)\n"),
			idxEnd - idxStart, idxStart + 1, idxEnd);
		logString.append(s);
	}

	if (idxEnd == 0) {
		idxEnd = pwd_count_total; //pwd_count;
	}

	if (bSaveWordList) {
		INT64 len = 0;
		for (i = 0; i < m_vpwdParts.size(); i++)
		{
			len += m_vpwdParts[i].charSet[0].charItem.size();
		}

		len += 2; //for line endings

		s.Printf(wxT("Estimated filesize:\n    %s\n"), FormatBytes(len * (idxEnd - idxStart)));
		logString.append(s);
	}

	if (!bSaveWordList) {
		if (keyFiles.size() > 0 && useKeyFiles) {
			if (firstKeyFile) {
				KeyFileRemoveAll(&firstKeyFile);
			}
			logString.append(wxT("Using key files:\n"));
			for (i = 0; i < keyFiles.size(); i++) {
				if (!canAccessFile((char *)keyFiles[i].c_str(), true, NULL)) {
					err.Printf(wxT("ERROR cannot read keyfile\n  %s\n"), keyFiles[i].c_str());
					goto END;
				}
				s.Printf(wxT("  %s\n"), keyFiles[i].c_str());
				logString.append(s);
				KeyFile *kf = (KeyFile *)malloc(sizeof(KeyFile));
				memcpy(kf->FileName, keyFiles[i].c_str(), sizeof(kf->FileName));
				firstKeyFile = KeyFileAdd(firstKeyFile, kf);
			}
		}

		cipher_set.cipher_count = 0;
		cipher_set.hash_count = 0;
		cipher_set.mode_count = 0;
		cipher_set.externalPKCS = externalPKCS;

		logString.append(wxT("Cipher:\n"));
		if (user_cipher.size() == 0)
		{
			for (i = 0; i < TC_CIPHER_COUNT + TC_MULTICIPHER_COUNT; i++) {
				s.Printf(wxT("  %s\n"), cipherNames[i]);
				logString.append(s);
			}
		}
		else
		{
			sort(user_cipher.begin(), user_cipher.end());
			if (user_cipher[0] == OTF_ALL_CIPHERS)
			{
				user_cipher.clear();
				for (i = 0; i < TC_CIPHER_COUNT + TC_MULTICIPHER_COUNT; i++)
					s.Printf(wxT("  %s\n"), cipherNames[i]);
					logString.append(s);
			} else {
				user_cipher.erase(unique(user_cipher.begin(), user_cipher.end()), user_cipher.end()); //Delete dupes

				for (i = 0; i < user_cipher.size(); i++) {
					if ((user_cipher[i] >= TC_CIPHER_COUNT + TC_MULTICIPHER_COUNT) || user_cipher[i] < 0) {
						s.Printf(wxT("  WARNING: Skipping illegal cipher value %d\n"), user_cipher[i]);
						logString.append(s);
						user_cipher.erase(user_cipher.begin() + i);
						i--;
						continue;
					}
					s.Printf(wxT("  %s\n"), cipherNames[user_cipher[i]]);
					logString.append(s);
				}
			}

			cipher_set.cipher_count = user_cipher.size();
			for (i = 0; i < user_cipher.size(); i++) {
				cipher_set.cipher[i] = user_cipher[i];
			}
		}

		logString.append(wxT("Hash:\n"));
		if (user_hash.size() == 0)
		{
			for (i = 0; i < TC_HASH_COUNT; i++) {
				s.Printf(wxT("  %s\n"), hashNames[i]);
				logString.append(s);
			}
		}
		else
		{
			sort(user_hash.begin(), user_hash.end());
			if (user_hash[0] == OTF_ALL_HASHES)
			{
				user_hash.clear();
				for (i = 0; i < TC_HASH_COUNT; i++)
					s.Printf(wxT("  %s\n"), hashNames[i]);
					logString.append(s);
			} else {
				user_hash.erase(unique(user_hash.begin(), user_hash.end()), user_hash.end()); //Delete dupes

				for (i = 0; i < user_hash.size(); i++) {
					if ((user_hash[i] >= TC_HASH_COUNT) || user_hash[i] < 0) {
						s.Printf(wxT("  WARNING: Skipping illegal hash value %d\n"), user_hash[i]);
						logString.append(s);
						user_hash.erase(user_hash.begin() + i);
						i--;
						continue;
					}
					s.Printf(wxT("  %s\n"), hashNames[user_hash[i]]);
					logString.append(s);
				}
			}

			cipher_set.hash_count = user_hash.size();
			for (i = 0; i < user_hash.size(); i++) {
				cipher_set.hash[i] = user_hash[i];
			}
		}

		logString.append(wxT("Mode:\n"));
		if (user_mode.size() == 0)
		{
			for (i = 0; i < TC_MODE_COUNT; i++) {
				s.Printf(wxT("  %s\n"), modeNames[i]);
				logString.append(s);
			}
		}
		else
		{
			sort(user_mode.begin(), user_mode.end());
			if (user_mode[0] == OTF_ALL_MODES)
			{
				user_mode.clear();
				for (i = 0; i < TC_MODE_COUNT; i++)
					s.Printf(wxT("  %s\n"), modeNames[i]);
					logString.append(s);
			} else {
				user_mode.erase(unique(user_mode.begin(), user_mode.end()), user_mode.end()); //Delete dupes

				for (i = 0; i < user_mode.size(); i++) {
					if ((user_mode[i] >= TC_MODE_COUNT) || user_mode[i] < 0) {
						s.Printf(wxT("  WARNING: Skipping illegal mode value %d\n"), user_mode[i]);
						logString.append(s);
						user_mode.erase(user_mode.begin() + i);
						i--;
						continue;
					}
					s.Printf(wxT("  %s\n"), modeNames[user_mode[i]]);
					logString.append(s);
				}
			}

			cipher_set.mode_count = user_mode.size();
			for (i = 0; i < user_mode.size(); i++) {
				cipher_set.mode[i] = user_mode[i];
			}
		}
		//
		/////////////////////////////////

		/////////////////////////////////
		//
		for (int i = 0; i < threads; i++) {
			hEvent[i] = CreateEvent(NULL, TRUE, TRUE, NULL);
			if (hEvent[i] == INVALID_HANDLE_VALUE) {
				err.Printf(wxT("Failed to create hEvent %d\n"), i);
				goto END;
			}
			hEventThread[i] = CreateEvent(NULL, TRUE, FALSE, NULL);
				if (hEventThread[i] == INVALID_HANDLE_VALUE) {
					err.Printf(wxT("Failed to create hEventThread %d\n"), i);
					goto END;
				}
			hThread[i] = (HANDLE) _beginthread(encryptionThread, 0, (void*)i);
				if (hThread[i] == INVALID_HANDLE_VALUE) {
					err.Printf(wxT("Failed to create hThread %d\n"), i);
					goto END;
				}

			thread_params[i].bRunning = FALSE;
		}
	}

	s.Printf(wxT("\n")
		wxT("%7s  ")
		wxT("%15s  ")
		wxT("%15s  ")
		wxT("%13s  ")
		wxT("%11s\n"),
		wxT("Percent"),
		wxT("Duration"),
		wxT("TimeLeft"),
		wxT("Speed"),
		wxT("Count")
	);
	logString.append(s);
	SendMessage(hwndMain, OTF_LOG, 0, 0);
	logString.clear();

	QueryPerformanceFrequency(&nFreq);
	QueryPerformanceCounter(&nStart);

	hEvent[threads] = CreateEvent(NULL, TRUE, TRUE, NULL);
	if (hEvent[threads] == INVALID_HANDLE_VALUE) {
		err.Printf(wxT("Failed to create password hEvent %d\n"), i);
		goto END;
	}
	hEventThread[threads] = CreateEvent(NULL, TRUE, FALSE, NULL);
		if (hEventThread[threads] == INVALID_HANDLE_VALUE) {
			err.Printf(wxT("Failed to create password hEventThread %d\n"), i);
			goto END;
		}

	if (dictionary) {
		pwd_count = 0; //Set to 0 so that pwd_pattern can't run

		hThread[threads] = (HANDLE) _beginthread(dictionaryThread, 0, (void*)0);
			if (hThread[threads] == INVALID_HANDLE_VALUE) {
				err.Printf(wxT("Failed to create password hThread %d\n"), i);
				goto END;
			}

		while (WAIT_OBJECT_0 != WaitForSingleObject(hEventThread[threads], 1000)) {
			QueryPerformanceCounter(&nEnd);
			double duration = 1.0 * (nEnd.QuadPart - nStart.QuadPart) / nFreq.QuadPart;
			double pps = (counter + counterSkipped) / duration;
			double timeLeft = pps == 0 ? 0 : (idxEnd - pwd_idx_dict) / pps;
			double percentage = ((double)(counter + counterSkipped) / (idxEnd - idxStart)) * 100.0;
			pps = counter / duration;

			timeFormatted(duration, (char *)&durationStr, 15);
			if (pps == 0) {
				memcpy(timeLeftStr, _T("???\0"), 4);
			} else {
				timeFormatted(timeLeft, (char *)&timeLeftStr, 15);
			}
			status.Printf(
				wxT("%6.0f%s  ")
				wxT("%15s  ") //"%12.2f  "
				wxT("%15s  ")
				wxT("%11.2f/s  ")
				wxT("%11I64d%s"),
				percentage, wxT("%"),
				durationStr, //timeFormatted(duration)
				timeLeftStr,
				pps,
				counter + counterSkipped, bVerbose ? wxT("\n") : wxT("\r")
			);
			PostMessage(hwndMain, OTF_STATUS, 0, 0);
		}
	}
	//
	/////////////////////////////////

	/////////////////////////////////
	// Subtract the fake print increment count
	actual_pwds_tried -= print_count;
	print_count = 0;

	if (pwd_count > 0) {
		// Make sure events are in the signaled state
		for (int i = 0; i < threads; i++) {
			SetEvent(hEvent[i]);
		}

		hThread[threads] = (HANDLE) _beginthread(passwordThread, 0, (void*)0);
			if (hThread[threads] == INVALID_HANDLE_VALUE) {
				err.Printf(wxT("Failed to create password hThread %d\n"), threads);
				goto END;
			}

		while (WAIT_OBJECT_0 != WaitForSingleObject(hEventThread[threads], 1000)) {
			QueryPerformanceCounter(&nEnd);
			double duration = 1.0 * (nEnd.QuadPart - nStart.QuadPart) / nFreq.QuadPart;
			double pps = (counter + counterSkipped) / duration;
			double timeLeft = pps == 0 ? 0 : (idxEnd + pwd_count - pwd_count_total - pwd_idx) / pps;
			double percentage = ((double)(counter + counterSkipped) / (idxEnd + pwd_count - pwd_count_total - idxStart)) * 100.0;
			pps = counter / duration;

			timeFormatted(duration, (char *)&durationStr, 15);
			if (pps == 0) {
				memcpy(timeLeftStr, _T("???\0"), 4);
			} else {
				timeFormatted(timeLeft, (char *)&timeLeftStr, 15);
			}
			status.Printf(
				wxT("%6.0f%s  ")
				wxT("%15s  ") //"%12.2f  "
				wxT("%15s  ")
				wxT("%11.2f/s  ")
				wxT("%11I64d%s"),
				percentage, wxT("%"),
				durationStr, //timeFormatted(duration)
				timeLeftStr,
				pps,
				counter + counterSkipped, bVerbose ? wxT("\n") : wxT("\r")
			);
			PostMessage(hwndMain, OTF_STATUS, 0, 0);
		}
	}
	//
	/////////////////////////////////

	/////////////////////////////////
	//
	QueryPerformanceCounter(&nEnd);

	dwEvent = WaitForMultipleObjects(
		threads,		// number of objects in array
		hEvent,			// array of objects
		TRUE,			// wait for all
		5000);
	// Subtract the fake print increment count
	actual_pwds_tried -= print_count;
	pwd_idx = actual_pwds_tried;

	bComplete = TRUE;

	double duration = 1.0 * (nEnd.QuadPart - nStart.QuadPart) / nFreq.QuadPart;
	double pps = counter / duration;
	double timeLeft = 0; //(idxEnd - pwd_idx) / pps;
	double percentage = ((double)(counter + counterSkipped) / (idxEnd + pwd_count - pwd_count_total - idxStart)) * 100.0;

	logString.clear();
	timeFormatted(duration, (char *)&durationStr, 12);
	timeFormatted(timeLeft, (char *)&timeLeftStr, 12);
	s.Printf(
		wxT("%6.0f%s  ")
		wxT("%15s  ") //"%12.2f  "
		wxT("%15s  ")
		wxT("%11.2f/s  ")
		wxT("%11I64d%s"),
		percentage, wxT("%"),
		durationStr, //timeFormatted(duration)
		timeLeftStr,
		pps,
		counter, wxT("\n")
	);
	if (counterSkipped > 0) {
		logString.append(s);
		s.Printf(
			wxT("                                       ")
			wxT("Skipped passwords  ")
			wxT("%11I64d%s")
			wxT("                                       ")
			wxT("  Total passwords  ")
			wxT("%11I64d%s"),
			counterSkipped, wxT("\n"),
			counter + counterSkipped, wxT("\n")
		);
	}

	if (hEvent[0]) {
		for (i = 0; i <= threads; i++) {
			if (hEvent[i]) {
				SetEvent(hEvent[i]);
				CloseHandle(hEvent[i]);
				hEvent[i] = NULL;
			}
			if (hEventThread[i]) {
				SetEvent(hEventThread[i]);
				CloseHandle(hEventThread[i]);
				hEventThread[i] = NULL;
			}
			//if (hThread[i]) {
			//	CloseHandle(hThread[i]);
			//	hThread[i] = NULL;
			//}
		}
	}

	SetEvent(hRunEvent);

	logString.append(s);
	SendMessage(hwndMain, OTF_LOG, 0, 0); //SendMessage PostMessage
	logString.clear();

	if (bSuccess && !bSaveWordList) {
		s.Printf(wxT("\nPassword Found\n"));
		logString.append(s);

		s.Printf(wxT("  %12s"), wxT("Header:"));
		logString.append(s);
		if (useBackupHeader) {
			logString.append(wxT(" Backup\n"));
		} else {
			logString.append(wxT(" Main\n"));
		}

		s.Printf(wxT("  %12s"), wxT("VolumeType:"));
		logString.append(s);
		switch (iVolType) {
			case OTF_VOL_TYPE_STD:
				s.Printf(wxT(" Standard"));
				break;
			case OTF_VOL_HIDDEN:
				s.Printf(wxT(" Hidden"));
				break;
			case OTF_VOL_BOOT:
				s.Printf(wxT(" System"));
				break;
			case OTF_VOL_HIDDEN_LEGACY1:
				s.Printf(wxT(" Hidden Legacy"));
				break;
			default:
				s.Printf(wxT(" Standard"));
				break;
		}
		logString.append(s);
		logString.append(wxT("\n"));

		s.Printf(wxT("  %12s %s\n"), wxT("password:"), found_pwd.c_str());
		logString.append(s);

		s.Printf(wxT("  %12s %s\n"), wxT("Cipher:"), cipherNames[ctx_cipher_idx]);
		logString.append(s);

		s.Printf(wxT("  %12s %s\n"), wxT("Hash:"), hashNames[ctx_hash_idx]);
		logString.append(s);

		s.Printf(wxT("  %12s %s\n"), wxT("Mode:"), modeNames[ctx_mode_idx]);
		logString.append(s);
	} else if (!bSaveWordList){
		s.Printf(wxT("\nPassword Not Found\n"));
		logString.append(s);
	} else if (bSaveWordList) {
		s.Printf(wxT("\nSaving Word List Complete\n"));
		logString.append(s);
	}

END:
	if (err.length()) {
		PostMessage(hwndMain, OTF_ERROR, 0, 0);
	}
	if (logString.length()) {
		PostMessage(hwndMain, OTF_LOG, 0, bSuccess ? 1 : 2);
	}

	if (hEvent[0]) {
		for (i = 0; i <= threads; i++) {
			if (hEvent[i]) {
				SetEvent(hEvent[i]);
				CloseHandle(hEvent[i]);
				hEvent[i] = NULL;
			}
			if (hEventThread[i]) {
				SetEvent(hEventThread[i]);
				CloseHandle(hEventThread[i]);
				hEventThread[i] = NULL;
			}
			//if (hThread[i]) {
			//	CloseHandle(hThread[i]);
			//	hThread[i] = NULL;
			//}
		}
	}

	SetEvent(hRunEvent);
}
